using System.Collections.Generic;
using UnityEditor;
using UnityEngine;
using UnityEngine.Events;

namespace PathTool
{
    public class WaypointsManager : MonoBehaviour
    {
        [Header("Waypoints Setup (auto-riempito dall'editor)")]
        public Waypoint[] waypoints;
        private int currentIndex = 0;
        [SerializeField] private UnityEvent OnTrackComplete;

        [Header("Debug Settings")]
        public Color pathColor = Color.cyan; // colore del percorso tracciato

        private void Start()
        {
            if (waypoints == null || waypoints.Length == 0) return;

            foreach (var wp in waypoints)
                wp.SetActive(false);

            waypoints[currentIndex].SetActive(true);
        }

        public void WaypointReached(Waypoint waypoint)
        {
            if (waypoint.index != currentIndex) return;

            Debug.Log("Waypoint " + waypoint.index + " raggiunto correttamente!");

            waypoints[currentIndex].SetActive(false);

            currentIndex++;
            if (currentIndex < waypoints.Length)
            {
                waypoints[currentIndex].SetActive(true);
            }
            else
            {
                Debug.Log("Percorso completato!");
                OnTrackComplete?.Invoke();
            }
        }

        private void OnDrawGizmos()
        {
            Gizmos.DrawSphere(transform.position, 0.5f);
            if (waypoints == null || waypoints.Length < 2) return;

            Gizmos.color = pathColor; // usa il colore scelto nellInspector

            for (int i = 0; i < waypoints.Length - 1; i++)
            {
                if (waypoints[i] != null && waypoints[i + 1] != null)
                {
                    Gizmos.DrawLine(waypoints[i].transform.position, waypoints[i + 1].transform.position);
                }
            }
        }

        public void DetectWaypoints()
        {
            // Trova tutti i waypoint nei figli
            var found = new List<Waypoint>(transform.GetComponentsInChildren<Waypoint>());
            if (found.Count == 0) return;

            List<Waypoint> ordered = new List<Waypoint>();

            // Punto di partenza: il pi vicino al manager
            Transform origin = transform;
            Waypoint current = GetClosestWaypoint(origin.position, found);
            ordered.Add(current);
            found.Remove(current);

            // Ciclo fino a esaurimento
            while (found.Count > 0)
            {
                current = GetClosestWaypoint(current.transform.position, found);
                ordered.Add(current);
                found.Remove(current);
            }

            // Aggiorna array
            waypoints = ordered.ToArray();

            // Aggiorna indici
            for (int i = 0; i < waypoints.Length; i++)
            {
                waypoints[i].index = i;
            }
        }

        private Waypoint GetClosestWaypoint(Vector3 from, List<Waypoint> list)
        {
            float minDist = Mathf.Infinity;
            Waypoint closest = null;

            foreach (var wp in list)
            {
                float distance = Vector3.Distance(from, wp.transform.position);
                if (distance < minDist)
                {
                    minDist = distance;
                    closest = wp;
                }
            }
            return closest;
        }
    }
}
