using UnityEngine;
using UnityEditor;
using UnityEngine.Splines;
using System.Collections.Generic;

namespace PathTool
{
    public class FlightPathTool : EditorWindow
    {
        private SplineContainer splineContainer;
        private GameObject targetPrefab;
        private int numberOfTargets = 10;
        private string newSplineName;
        private int distance = 5;

        private void OnEnable()
        {
            targetPrefab = LoadPrefabFromPackage();
        }

        [MenuItem("Advepa/Flight Path Tool")]
        public static void ShowWindow()
        {
            GetWindow<FlightPathTool>("Flight Path Tool");
        }

        private void OnGUI()
        {
            GUILayout.Label("Flight Path Setup", EditorStyles.boldLabel);

            splineContainer = (SplineContainer)EditorGUILayout.ObjectField("Spline Container", splineContainer, typeof(SplineContainer), true);
            targetPrefab = (GameObject)EditorGUILayout.ObjectField("Target Prefab", targetPrefab, typeof(GameObject), false);
            numberOfTargets = EditorGUILayout.IntSlider("Number of Targets", numberOfTargets, 2, 100);

            if (GUILayout.Button("Generate new Spline"))
            {
                if (splineContainer != null)
                {
                    splineContainer = null;
                }

                StringPopupWindow.Show("Test", result =>
                {
                    newSplineName = result;
                    GenerateSpline();
                });

            }
            if (GUILayout.Button("Generate Targets"))
            {
                if (splineContainer != null && targetPrefab != null)
                {
                    GenerateTargets();
                }
                else
                {
                    Debug.LogWarning("Assign Spline Container and Target Prefab first!");
                }
            }

            if (GUILayout.Button("Clear Targets"))
            {
                if (splineContainer != null)
                    ClearTargets();
                else
                    Debug.LogWarning("Assign Spline Container and Target Prefab first!");
            }


            if (Selection.activeGameObject != null)
            {
                if (Selection.activeGameObject.GetComponent<WaypointsManager>() != null)
                    splineContainer = Selection.activeGameObject.GetComponent<SplineContainer>();
            }
        }

        private void GenerateSpline()
        {
            GameObject spline = new GameObject();
            spline.name = newSplineName;
            spline.gameObject.AddComponent<WaypointsManager>();
            spline.AddComponent<SplineContainer>();
            Camera sceneCam = SceneView.lastActiveSceneView.camera;
            Vector3 spawnPos = sceneCam.transform.position + sceneCam.transform.forward * distance;
            spline.transform.position = spawnPos;
            splineContainer = spline.GetComponent<SplineContainer>();
            Undo.RegisterCreatedObjectUndo(splineContainer, "Spawn Object In Front of Scene Camera");
            splineContainer.Spline.Add(new Vector3(0, 0, 0));
            splineContainer.Spline.Add(new Vector3(2, 0, 2));
            splineContainer.Spline.Add(new Vector3(0, 0, 4));
            Selection.activeGameObject = splineContainer.gameObject;
        }

        private void GenerateTargets()
        {
            ClearTargets();

            Spline spline = splineContainer.Spline;

            for (int i = 0; i < spline.Count; i++)
            {
                float t = (float)i / (spline.Count - 1);

                // Ottieni posizione e tangente lungo la spline
                Vector3 pos = spline[i].Position;
                Vector3 tangent = spline.EvaluateTangent(t);

                // Instanzia prefab
                GameObject newTarget = (GameObject)PrefabUtility.InstantiatePrefab(targetPrefab);
                newTarget.transform.SetParent(splineContainer.transform, false);

                // Set posizione locale (cos segue la splineContainer se la muovi)
                newTarget.transform.localPosition = pos;

                // Rotazione: forward = tangente della spline, up = asse Y
                newTarget.transform.localRotation = Quaternion.LookRotation(tangent, Vector3.up);

                newTarget.name = "Target_" + i;
            }
            splineContainer.GetComponent<WaypointsManager>().DetectWaypoints();
        }


        private void ClearTargets()
        {
            List<GameObject> toDestroy = new List<GameObject>();
            foreach (Transform child in splineContainer.transform)
            {
                if (PrefabUtility.GetPrefabAssetType(child.gameObject) != PrefabAssetType.NotAPrefab)
                {
                    toDestroy.Add(child.gameObject);
                }
            }

            foreach (var obj in toDestroy)
            {
                DestroyImmediate(obj);
            }
        }

        public static GameObject LoadPrefabFromPackage()
        {
            // Percorso relativo al package
            string path = "Packages/com.advepa.flightpath/Runtime/Prefabs/WayPointVertical.prefab";

            GameObject prefab = AssetDatabase.LoadAssetAtPath<GameObject>(path);

            if (prefab == null)
                Debug.LogError($"Prefab non trovato al percorso: {path}");

            return prefab;
        }
    }
}
